/* 
 * upe.c
 *
 * Моделює роботу служби "Луна" на базі неіменованих каналів
 * (із зворотним зв'язком між сервером і клієнтом).
 * Сервером виступає дочірній процес, клієнтом - батьківський.
 * Клієнт запитує в користувача повідомлення і передає їх серверу.
 * Сервер отримані повідомлення повертає клієнту. Клієнт отримані від 
 * сервера відповіді виводить у стандартний потік виведення результатів.
 * Ілюструє порядок застосування неіменованих каналів.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>


int main()
{
        enum { BUFSIZE = 80 };
        char buf[BUFSIZE];		
        /* Дескриптори каналу для передачі повідомлень від клієнта
           серверу */
        int pipe_to_server[2];
        /* Дескриптори каналу для передачі повідомлень від сервера
           клієнту */
        int pipe_from_server[2];
        pid_t server_pid;

        /* Створює неіменований канал для передачі повідомлень від
           клієнта серверу. */
        if (pipe(pipe_to_server) != 0) {
                perror("pipe(pipe_to_server)");
                exit(EXIT_FAILURE);
        }	
        /* Створює неіменований канал для передачі повідомлень від
           сервера клієнту. */
        if (pipe(pipe_from_server) != 0) {
                perror("pipe(pipe_from_server)");
                exit(EXIT_FAILURE);
        }	

        /* Створює дочірній процес. */
        server_pid = fork();
        switch (server_pid) {
        case -1:
                /* Помилка створення дочірнього процесу */
                perror("fork()");
                exit(EXIT_FAILURE);
        case 0:
                /* Дочірній процес (сервер) */
                /* Закриває зайві дескриптори каналів. */
                if (close(pipe_to_server[1]) != 0)
                        perror("close(pipe_to_server[1]");
                if (close(pipe_from_server[0]) != 0)
                        perror("close(pipe_to_server[1]");
                while (1) {
                        int count;

                        /* Отримує повідомлення. */
                        count = read(pipe_to_server[0], buf, BUFSIZE);
                        if (count < 0) {
                                perror("read(pipe_to_server[0], ...)");
                                exit(EXIT_FAILURE);
                        }
                        if (count == 0)
                                /* Клієнт закрив канал. */
                                break;
                        /* Повертає повідомлення клієнту. */
                        if (write(pipe_from_server[1], buf, BUFSIZE) < 0) {
                                perror("write(pipe_from_server[1], ...)");
                                exit(EXIT_FAILURE);
                        }
                }
                exit(EXIT_SUCCESS);
        default:
                /* Батьківський процес (клієнт) */
                /* Закриває зайві дескриптори каналів. */
                if (close(pipe_to_server[0]) != 0)
                        perror("close(pipe_to_server[0]");
                if (close(pipe_from_server[1]) != 0)
                        perror("close(pipe_from_server[1]");
                do {
                        /* Запитує в користувача повідомлення. */
                        printf("Request:  ");
                        /* Отримує від користувача повідомлення. */
                        scanf("%s", buf);
                        /* Передає повідомлення серверу. */
                        if (write(pipe_to_server[1], buf, BUFSIZE) < 0) {
                                perror("write(pipe_to_server[1], ...)");
                                exit(EXIT_FAILURE);
                        }
                        buf[0] = '\0';
                        /* Отримує відповідь сервера. */
                        if (read(pipe_from_server[0], buf, BUFSIZE) < 0) {
                                perror("read(pipe_from_server[0], ...)");
                                exit(EXIT_FAILURE);
                        }
                        /* Виводить відповідь сервера. */
                        printf("Answer:  %s\n", buf);
                        /* Запитує в користувача продовження роботи. */
                        printf("Another request? [y/n] ");
                        scanf("%s", buf);
                } while (!(strcmp(buf, "y") * strcmp(buf, "Y")));	
                exit(EXIT_SUCCESS);
        }
}
